<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ContactMessage;
use Illuminate\Http\Request;

class ContactMessageController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = ContactMessage::query()->orderBy('created_at', 'desc');
        
        // Filter by read status
        if ($request->has('status')) {
            if ($request->status === 'unread') {
                $query->unread();
            } elseif ($request->status === 'read') {
                $query->read();
            }
        }

        $messages = $query->paginate(20);
        $unreadCount = ContactMessage::unread()->count();

        return view('admin.contact-messages.index', compact('messages', 'unreadCount'));
    }

    /**
     * Display the specified resource.
     */
    public function show(ContactMessage $contactMessage)
    {
        // Mark as read when viewed
        if (!$contactMessage->is_read) {
            $contactMessage->markAsRead();
        }

        return view('admin.contact-messages.show', compact('contactMessage'));
    }

    /**
     * Mark message as read/unread
     */
    public function toggleRead(ContactMessage $contactMessage)
    {
        if ($contactMessage->is_read) {
            $contactMessage->update([
                'is_read' => false,
                'read_at' => null
            ]);
            $status = 'unread';
        } else {
            $contactMessage->markAsRead();
            $status = 'read';
        }

        return redirect()->back()->with('success', "Message marked as {$status}.");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ContactMessage $contactMessage)
    {
        $contactMessage->delete();

        return redirect()->route('admin.contact-messages.index')
                        ->with('success', 'Message deleted successfully.');
    }

    /**
     * Mark all messages as read
     */
    public function markAllRead()
    {
        ContactMessage::unread()->update([
            'is_read' => true,
            'read_at' => now()
        ]);

        return redirect()->back()->with('success', 'All messages marked as read.');
    }
}
