@extends('layouts.admin')

@section('title', 'Edit Skill')

@section('content')
<div class="d-flex justify-content-between align-items-center mb-4">
    <h1 class="h3">Edit Skill</h1>
    <a href="{{ route('admin.skills.index') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-left"></i> Back to Skills
    </a>
</div>

<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-body">
                <form action="{{ route('admin.skills.update', $skill) }}" method="POST">
                    @csrf
                    @method('PUT')
                    
                    <div class="mb-3">
                        <label for="name" class="form-label">Skill Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control @error('name') is-invalid @enderror" 
                               id="name" name="name" value="{{ old('name', $skill->name) }}" required>
                        @error('name')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">e.g., Photography, Web Design, Photoshop</div>
                    </div>

                    <div class="mb-3">
                        <label for="percentage" class="form-label">Proficiency Level <span class="text-danger">*</span></label>
                        <div class="input-group">
                            <input type="range" class="form-range" id="percentage-range" min="0" max="100" value="{{ old('percentage', $skill->percentage) }}" oninput="updatePercentage(this.value)">
                            <input type="number" class="form-control @error('percentage') is-invalid @enderror" 
                                   id="percentage" name="percentage" value="{{ old('percentage', $skill->percentage) }}" min="0" max="100" required style="max-width: 100px;" oninput="updateRange(this.value)">
                            <span class="input-group-text">%</span>
                        </div>
                        @error('percentage')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">Your proficiency level from 0% to 100%</div>
                    </div>

                    <div class="mb-3">
                        <label for="color" class="form-label">Progress Bar Color <span class="text-danger">*</span></label>
                        <div class="input-group">
                            <input type="color" class="form-control form-control-color @error('color') is-invalid @enderror" 
                                   id="color" name="color" value="{{ old('color', $skill->color) }}" required style="max-width: 60px;">
                            <input type="text" class="form-control" id="color-hex" value="{{ old('color', $skill->color) }}" readonly>
                        </div>
                        @error('color')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">Choose a color for the progress bar</div>
                    </div>

                    <div class="mb-3">
                        <label for="sort_order" class="form-label">Sort Order</label>
                        <input type="number" class="form-control @error('sort_order') is-invalid @enderror" 
                               id="sort_order" name="sort_order" value="{{ old('sort_order', $skill->sort_order) }}">
                        @error('sort_order')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">Lower numbers appear first</div>
                    </div>

                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="is_active" name="is_active" value="1" {{ old('is_active', $skill->is_active) ? 'checked' : '' }}>
                            <label class="form-check-label" for="is_active">
                                Active (show on website)
                            </label>
                        </div>
                    </div>

                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Update Skill
                        </button>
                        <a href="{{ route('admin.skills.index') }}" class="btn btn-secondary">Cancel</a>
                        <a href="{{ route('admin.skills.show', $skill) }}" class="btn btn-info">
                            <i class="fas fa-eye"></i> View
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Current Skill Preview</h5>
            </div>
            <div class="card-body">
                <div id="current-preview" class="mb-3">
                    <p><strong>{{ $skill->name }}</strong></p>
                    <div class="progress" style="height: 25px;">
                        <div class="progress-bar" role="progressbar" style="width: {{ $skill->percentage }}%; background-color: {{ $skill->color }};">
                            <span>{{ $skill->percentage }}%</span>
                        </div>
                    </div>
                </div>
                
                <hr>
                
                <div id="skill-preview">
                    <h6>Live Preview:</h6>
                    <p id="preview-name">{{ $skill->name }}</p>
                    <div class="progress" style="height: 25px;">
                        <div class="progress-bar" role="progressbar" id="preview-bar" style="width: {{ $skill->percentage }}%; background-color: {{ $skill->color }};">
                            <span id="preview-percentage">{{ $skill->percentage }}%</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="card mt-3">
            <div class="card-header">
                <h5 class="card-title mb-0">Skill Details</h5>
            </div>
            <div class="card-body">
                <p><strong>Created:</strong> {{ $skill->created_at->format('F j, Y \a\t g:i A') }}</p>
                <p><strong>Updated:</strong> {{ $skill->updated_at->format('F j, Y \a\t g:i A') }}</p>
                <p><strong>Status:</strong> 
                    @if($skill->is_active)
                        <span class="badge bg-success">Active</span>
                    @else
                        <span class="badge bg-danger">Inactive</span>
                    @endif
                </p>
                <p><strong>Proficiency Level:</strong>
                    @if($skill->percentage >= 91)
                        <span class="badge bg-success">Expert</span>
                    @elseif($skill->percentage >= 76)
                        <span class="badge bg-info">Advanced</span>
                    @elseif($skill->percentage >= 51)
                        <span class="badge bg-warning">Intermediate</span>
                    @elseif($skill->percentage >= 26)
                        <span class="badge bg-secondary">Basic</span>
                    @else
                        <span class="badge bg-light text-dark">Beginner</span>
                    @endif
                </p>
            </div>
        </div>

        <div class="card mt-3">
            <div class="card-header">
                <h5 class="card-title mb-0">Color Presets</h5>
            </div>
            <div class="card-body">
                <div class="d-flex flex-wrap gap-2">
                    <button type="button" class="btn btn-sm" style="background-color: #007bff; width: 30px; height: 30px;" onclick="setColor('#007bff')"></button>
                    <button type="button" class="btn btn-sm" style="background-color: #28a745; width: 30px; height: 30px;" onclick="setColor('#28a745')"></button>
                    <button type="button" class="btn btn-sm" style="background-color: #ffc107; width: 30px; height: 30px;" onclick="setColor('#ffc107')"></button>
                    <button type="button" class="btn btn-sm" style="background-color: #dc3545; width: 30px; height: 30px;" onclick="setColor('#dc3545')"></button>
                    <button type="button" class="btn btn-sm" style="background-color: #6f42c1; width: 30px; height: 30px;" onclick="setColor('#6f42c1')"></button>
                    <button type="button" class="btn btn-sm" style="background-color: #fd7e14; width: 30px; height: 30px;" onclick="setColor('#fd7e14')"></button>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
function updatePercentage(value) {
    document.getElementById('percentage').value = value;
    updatePreview();
}

function updateRange(value) {
    document.getElementById('percentage-range').value = value;
    updatePreview();
}

function updatePreview() {
    const name = document.getElementById('name').value || 'Skill Name';
    const percentage = document.getElementById('percentage').value;
    const color = document.getElementById('color').value;
    
    document.getElementById('preview-name').textContent = name;
    document.getElementById('preview-bar').style.width = percentage + '%';
    document.getElementById('preview-bar').style.backgroundColor = color;
    document.getElementById('preview-percentage').textContent = percentage + '%';
}

function setColor(color) {
    document.getElementById('color').value = color;
    document.getElementById('color-hex').value = color;
    updatePreview();
}

// Update color hex display when color picker changes
document.getElementById('color').addEventListener('change', function() {
    document.getElementById('color-hex').value = this.value;
    updatePreview();
});

// Update preview when name changes
document.getElementById('name').addEventListener('input', updatePreview);

// Initial preview update
updatePreview();
</script>
@endsection
